
"""
------------------------------------
DISEASE DYNAMICS
------------------------------------

"""
import numpy as np
import matplotlib.pyplot as plt
from scipy.integrate import solve_ivp

def model_ode(t, y, u, alpha, beta, gamma, mu, D, p_matrix, population, population_eff):
    """
    Defines a system of ordinary differential equations (ODEs) for a compartmental model in epidemiology.

    Parameters:
    t (float): The current time.
    y (array-like): The current state of the system. Should be a 1D array of length 4*K,
                    where the first K elements represent the susceptible individuals,
                    the next K elements represent the exposed individuals,
                    the next K elements represent the infectious individuals,
                    and the last K elements represent the recovered individuals.
    u (array-like): The vaccination rate for each city.
                    Should be a 1D array of length K.
    alpha (float): Proportion of the night among the 24 hours.
    beta (array-like): The transmission rate for each city.
                       Should be a 1D array of length K.
    gamma (float): The recovery rate (the inverse of the infectious period).
    mu (float): The birth and natural death rate.
    K (int): The number of cities or groups in the population.
    p_matrix (array-like): A matrix that represents the transitions between different cities.
                           Should be a 2D array of shape (K, K).
    population (array-like): The total population in each city.
                             Should be a 1D array of length K.
    population_eff (array-like): The effective population during the day.
                                 Should be a 1D array of length K.
                                 P_i^{\mathrm{eff}} = \sum_{j=1}^K p_matrix{ji} population_j

    Returns:
    array-like: The rates of change of the susceptible, exposed, infectious, and recovered individuals.
                Returns a 1D array of length 4*K.
    """
    S, I, R, _ = y[:D], y[D:2*D], y[2*D:3*D], y[3*D:]
    I_eff = (I * population) @ p_matrix / population_eff
    S_dot = -alpha * beta * S * I - (1-alpha) * S * (p_matrix @ (beta * I_eff))
    I_dot = -S_dot - (gamma + mu) * I
    R_dot = u * S + gamma * I - mu * R
    H_dot = -S_dot
    S_dot += mu - (u + mu) * S

    return np.hstack([S_dot, I_dot, R_dot, H_dot])

 # Parameters

D = 10 #Number of identical districts

populations = np.array([1e5] * D)
p=5e-3 #fraction of time that individuals spend in each neighboring distric
r=1
M=np.zeros((D, D), dtype=float)
for i in range(D):
  M[i,i] = 1
  for j in range(i+1,i+r+1):
    if j<D :
      M[i,j] = p
      M[i , i] -= M[i,j]
  for j in range(i-r,i):
    if j>=0 :
      M[i,j] = p
      M[i , i] -= M[i,j]

p_matrix=M;

population_eff = populations @ p_matrix

beta=1.0
beta_arr = np.array([beta] * D)
alpha = 0
gamma = 0.5
mu = 0

# Initial condition
y0 = np.zeros(4*D)
y0[D] = D/1e6
y0[0:D] = 1-y0[D:2*D]

n_days = 100
t_eval = np.linspace(0, n_days, 20 * n_days)

args = (np.zeros(D), alpha, beta_arr, gamma, mu, D, p_matrix, populations, population_eff)

sol = solve_ivp(fun=lambda t, y: model_ode(t, y, *args),
                t_span=(0,n_days),
                y0=y0,
                t_eval=t_eval,
                rtol=1e-9,
                atol=1e-12,
                max_step=1e-2
            )

tnetwork = sol.t
ynetwork = sol.y

plt.figure(figsize=(10, 6))

Itot = np.zeros_like(tnetwork)
for i in range(D):
    plt.plot(sol.t, sol.y[D+i], label=f'Infected Pop. {i+1}')
    Itot += sol.y[D + i]/D  # Sum all I_i(t)
plt.plot(sol.t, Itot * D, label=f'Total Infected Population')
plt.xlabel('Time (days)')
plt.ylabel('Population')
plt.title('Disease Dynamics')
plt.legend()
plt.grid(True)
plt.savefig("metapop.svg")
plt.show()

"""
------------------------------------
# AGGREGATED MODEL
------------------------------------

"""

import numpy as np
import matplotlib.pyplot as plt
from scipy.integrate import odeint
from scipy.optimize import root_scalar


def pSIR_model(x,t,rho,betar,alpha):
    pI, pR, S, I, R = x

    pS = N - pI - pR

    dpR_dt = betar/Sinf *R*S/pI;
    dpS_dt = -rho * I *np.arctan(alpha * pS/N * np.pi/2) * 2/np.pi
    dpI_dt = -dpR_dt-dpS_dt;

    dS_dt = -dpS_dt- beta * I * S/pI - Sinf*dpR_dt;
    dI_dt = beta * I * S / pI - gamma * I;
    dR_dt = gamma*I - Rinf * dpR_dt;
    return [dpI_dt,dpR_dt,dS_dt, dI_dt, dR_dt]


def eqRinf(Rinf):
        return Rinf - (1 - np.exp(-beta/gamma * Rinf))


# Parameters
rho = 0.657   # Propagation rate
betar=0.152
alpha = 32.81

beta = 1  # Infection rate
gamma = 0.5 # Recovery rate
N = 1    # Total population
Rinfresult = root_scalar(eqRinf, x0=0.5, method='newton', bracket=[0, 1])
Rinf=Rinfresult.root
Sinf=1-Rinf

# Initial conditions
pI_0 = 0.105   # Initial propagation rate
I_0 = 1/1e6     # Initial infected population
S_0 = pI_0-I_0  # Initial susceptible population (exposed)
R_0 = 0      # Initial recovered population
pR_0 = 0
x0 = [pI_0, pR_0, S_0, I_0, R_0]

# Time solution
t_start = 0.0
t_end = 100.0
num_points = 1000
time_points = tnetwork

#solution
sol = odeint(pSIR_model, x0, time_points,rtol=1e-6, atol=1e-9,args=(rho,betar,alpha))

I = sol[:, 3]

plt.figure(figsize=(10, 6))
plt.plot(time_points, I, label='Infected (I)')
plt.plot(time_points, Itot, label='Network Infected (Itot)')
plt.title('Aggregated Model')
plt.xlabel('Time')
plt.ylabel('Infected')
plt.grid(True)
plt.legend()
plt.show()



